/*
 * Copyright (c) 2015-2020 Sinch AB. All rights reserved.
 *
 * See LICENSE file for license terms and information.
 */

#import <Foundation/Foundation.h>

#import <SinchRTC/SINExport.h>

#import <SinchRTC/SINClient.h>
#import <SinchRTC/SINClientRegistration.h>

#import <SinchRTC/SINCallClient.h>
#import <SinchRTC/SINCall.h>
#import <SinchRTC/SINCallDetails.h>
#import <SinchRTC/SINIceCandidateTransportType.h>
#import <SinchRTC/SINWebRtcCallConfiguration.h>

#import <SinchRTC/SINAudioController.h>

#import <SinchRTC/SINVideoController.h>
#import <SinchRTC/SINUIView+Fullscreen.h>
#import <SinchRTC/SINVideoFrameCallback.h>

#import <SinchRTC/SINManagedPush.h>
#import <SinchRTC/SINAPSEnvironment.h>

#import <SinchRTC/SINNotificationResult.h>
#import <SinchRTC/SINCallNotificationResult.h>

#import <SinchRTC/SINLogSeverity.h>
#import <SinchRTC/SINLogCallback.h>
#import <SinchRTC/SINError.h>

#import <SinchRTC/SINCallQualityWarning.h>

/**
 * The Sinch class is used to instantiate a SINClient.
 *
 * This is the starting point for an app that wishes to use the Sinch SDK.
 *
 * To construct a SINClient, the required configuration parameters are:
 *
 *  - Application Key
 *  - Environment host (default is 'ocra.api.sinch.com')
 *  - UserID
 *
 * It is optional to specify:
 *
 *  - CLI (Calling-Line Identifier / Caller-ID) that will be used for calls
 *    terminated to PSTN (Publicly Switched Telephone Network).
 */
SIN_EXPORT
@interface Sinch : NSObject

#pragma mark - Basic factory methods

// Sinch should not be used with alloc/init. Use the available class factory methods instead.
- (instancetype)init NS_UNAVAILABLE;

/**
 * Instantiate a new client.
 * The creation of the client may fail, as a result of input parameters being nil or not matching length criteria,
 * or because of internal failure when creating application support directory.
 * If any failure occurs, the returned value will be nil, and `error` parameter will contain relevant information.
 *
 * @return The newly instantiated client.
 *
 * @param applicationKey Application Key identifying the application.
 *
 * @param environmentHost Host for base URL for the Sinch RTC API environment to be used. E.g. 'ocra.api.sinch.com'
 *
 * @param userId ID of the local user.
 *
 * @param error Error object that describes the problem in case the method returns NO. It can
 *              be nil.
 *
 * @return a SINClient object, or nil if creation of SINClient failed.
 *         If return value is nil, the value of error will contain more specific info about
 *         the failure.
 *
 * @see `SINClient`
 * @see `SINClientRegistration`
 */

+ (id<SINClient>)clientWithApplicationKey:(NSString *)applicationKey
                          environmentHost:(NSString *)environmentHost
                                   userId:(NSString *)userId
                                    error:(NSError **)error;

#pragma mark - Factory methods with support for CLI / PSTN

/**
 * Instantiate a new client with a CLI (may be used for PSTN-terminated calls).
 * The creation of the client may fail, as a result of input parameters being nil or not matching length criteria,
 * or because of internal failure when creating application support directory.
 * If any failure occurs, the returned value will be nil, and `error` parameter will contain relevant information.
 *
 * @return The newly instantiated client.
 *
 * @param applicationKey Application key identifying the application.
 *
 * @param environmentHost Host for base URL for the Sinch RTC API environment to be used. E.g. 'ocra.api.sinch.com'
 *
 * @param userId ID of the local user.
 *
 * @param cli Caller-ID when terminating calls to PSTN. Must be a valid phone number.
 *
 * @param error Error object that describes the problem in case the method returns NO. It can
 *              be nil.
 *
 * @return a SINClient object, or nil if creation of SINClient failed.
 *         If return value is nil, the value of error will contain more specific info about
 *         the failure.
 *
 * @see `SINClient`
 * @see `SINClientRegistration`
 */

+ (id<SINClient>)clientWithApplicationKey:(NSString *)applicationKey
                          environmentHost:(NSString *)environmentHost
                                   userId:(NSString *)userId
                                      cli:(NSString *)cli
                                    error:(NSError **)error;

#pragma mark - Push Notifications

/**
 * Instantiate a new `SINManagedPush` instance to enable Push Notifications
 * managed by the Sinch SDK and platform. When using managed push notifications,
 * push notifications will be sent by the Sinch platform provided that Apple
 * Push Notification Certificates for your application have been uploaded to Sinch.
 *
 * @param apsEnvironment Specification of which Apple Push Notification Service environment
 *                       the application is bound to (via code signing and Provisioning Profile).
 *
 * @see `SINAPSEnvironment`
 */
+ (SINManagedPush *)managedPushWithAPSEnvironment:(SINAPSEnvironment)apsEnvironment;

#pragma mark - Miscellaneous

/**
 * Set a log callback block.
 *
 * The Sinch SDK will emit all it's logging by invoking the specified block.
 *
 * - Important: Only log messages with severity level `SINLogSeverityWarn`
 * or higher to the console in release builds, to avoid flooding the
 * device console with debugging messages.
 *
 * @param block log callback block. IMPORTANT: The block may be invoked on any thread / GCD queue.
 */
+ (void)setLogCallback:(SINLogCallback)block;

/**
 * Specify the data protection type (NSFileProtectionType) for the files created and used by the Sinch SDK.
 * If not set specifically, the files will inherit the data protection level defined in your Application.
 *
 * Method should be called before creation any instances of Sinch SDK classes, e.g. `SINClient`, `SINManagedPush` etc.
 *
 * @param type the data protection type applied to the files created by the Sinch SDK.
 */
+ (void)setDataProtectionType:(NSFileProtectionType)type;

/**
 * Set a callback queue.
 *
 * The Sinch SDK will emit all callbacks of SINClientDelegate, SINCallClientDelegate, SINCallDelegate,
 * SINAudioControllerDelegate on the dispatch_queue specified with this method.
 *
 * Note that SINManagedPush delegate callbacks will still occur on any thread, because incoming VoIP
 * pushes have to be reported to CallKit synchronously.
 *
 * When not explicitly set, the callback queue defaults to the main queue.
 *
 * @param queue the dispatch_queue where callbacks will be invoked.
 *
 */
+ (void)setCallbackQueue:(dispatch_queue_t)queue;

/**
 * @return the callback queue set via +[Sinch setCallbackQueue:], or the main queue if it hasn't been
 * set explicitly.
 */
+ (dispatch_queue_t)getCallbackQueue;

/**
 * @return Sinch SDK version.
 */
+ (NSString *)versionString;

@end
