#import <Foundation/Foundation.h>
#import <CallKit/CallKit.h>
#import "SINClientMediator+Private.h"

@implementation SINClientMediator (CXProviderDelegate)

- (void)providerDidReset:(nonnull CXProvider *)provider {
  os_log(self.customLog, "CXProviderDelegate.providerDidReset()");

  /*
   End any ongoing calls if the provider resets, and remove them from the app's list of calls
   because they are no longer valid.
   */
  NSArray *calls = [self.callRegistry activeSinchCalls];
  [calls enumerateObjectsUsingBlock:^(id<SINCall> _Nonnull call, NSUInteger idx, BOOL *_Nonnull stop) {
    [call hangup];
  }];

  // Remove all calls from the app's list of calls.
  [self.callRegistry reset];
}

- (void)provider:(CXProvider *)provider didActivateAudioSession:(AVAudioSession *)audioSession {
  if (nil == self.client) {
    os_log_error(self.customLog,
                 "SinchClient not assigned when audio session is activating (provider:didActivate audioSession)");
    return;
  }
  [self.client.callClient didActivateAudioSession:audioSession];
}

- (void)provider:(CXProvider *)provider didDeactivateAudioSession:(AVAudioSession *)audioSession {
  if (nil == self.client) {
    os_log_error(self.customLog,
                 "SinchClient not assigned when audio session is activating (provider:didDeactivate audioSession)");
    return;
  }
  [self.client.callClient didDeactivateAudioSession:audioSession];
}

- (void)provider:(CXProvider *)provider performStartCallAction:(CXStartCallAction *)action {
  CallStartedCompletion callback = self.callStartedCallback;
  self.callStartedCallback = nil;

  if (self.client == nil) {
    os_log_error(self.customLog, "SinchClient not assigned when CXStartCallAction. Failing action");
    [action fail];
    if (callback) {
      NSError *error =
          [NSError errorWithDomain:@"com.app.domain"
                              code:1  // Code is just an example.
                          userInfo:@{NSLocalizedDescriptionKey : @"SinchClient not assigned when CXStartCallAction."}];
      callback(nil, error);
    }
    return;
  }

  os_log(self.customLog, "provider perform action: CXStartCallAction");

  id<SINCall> call = nil;
  NSError *error = nil;

  if (self.isVideo) {
    call = [self.client.callClient callUserVideoWithId:action.handle.value];
  } else {
    call = [self.client.callClient callUserWithId:action.handle.value];
  }

  if (call.state == SINCallStateEnded) {
    os_log(self.customLog, "Failed to create outgoing call, error: %@", [call.details.error localizedDescription]);
    [action fail];
    error = call.details.error;
  } else {
    [self.callRegistry addSinchCall:call];
    [self.callRegistry mapCallKitUUID:action.callUUID toSinchCallId:call.callId];
    // NOTE: Mediator is the only delegate of the call and will do the fanout of events
    call.delegate = self;

    [action fulfill];
  }
  if (callback) {
    callback(call, error);
  }
}

- (void)provider:(CXProvider *)provider performAnswerCallAction:(CXAnswerCallAction *)action {
  if (self.client == nil) {
    os_log_error(self.customLog, "SinchClient not assigned when CXAnswerCallAction. Failing action");
    [action fail];
    return;
  }

  id<SINCall> call = [self.callRegistry sinchCallForCallKitUUID:action.callUUID];
  if (call == nil) {
    [action fail];
    return;
  }

  os_log(self.customLog, "provider perform action: CXAnswerCallAction: %{public}@", call.callId);

  [self.client.audioController configureAudioSessionForCallKitCall];

  [call answer];

  [action fulfill];
}

- (void)provider:(CXProvider *)provider performEndCallAction:(CXEndCallAction *)action {
  if (self.client == nil) {
    os_log_error(self.customLog, "SinchClient not assigned when CXEndCallAction. Failing action");
    [action fail];
    return;
  }

  id<SINCall> call = [self.callRegistry sinchCallForCallKitUUID:action.callUUID];
  if (call == nil) {
    [action fail];
    return;
  }

  os_log(self.customLog, "provider perform action: CXEndCallAction: %{public}@", call.callId);

  [call hangup];

  [action fulfill];
}

- (void)provider:(CXProvider *)provider performSetMutedCallAction:(CXSetMutedCallAction *)action {
  if (self.client == nil) {
    os_log_error(self.customLog, "SinchClient not assigned when CXSetMutedCallAction. Failing action");
    [action fail];
    return;
  }

  os_log(self.customLog, "provider perform action: CXSetMutedCallAction, uuid: %{public}@",
         action.callUUID.description);

  if (self.acDelegate.muted) {
    [self.client.audioController unmute];
  } else {
    [self.client.audioController mute];
  }

  [action fulfill];
}

@end
