#import "CallViewController.h"
#import "AppDelegate.h"
#import "CallViewController+UI.h"
#import "SINClientMediator.h"

@interface CallViewController () <SINClientMediatorObserver>
@property (nonatomic, strong, readonly) id<SINClient> client;
@end

@implementation CallViewController

- (id<SINClient>)client {
  return self.callKitMediator.client;
}

- (void)setCallKitMediator:(SINClientMediator *)callKitMediator {
  _callKitMediator = callKitMediator;
  [_callKitMediator addObserver:self];
}

#pragma mark - UIViewController Cycle

- (void)viewDidLoad {
  [super viewDidLoad];

  NSAssert(self.callKitMediator != nil, @"self.callKitMediator must not be nil here", @"");
  NSAssert(self.callKitMediator.client != nil, @"self.callKitMediator.client must not be nil here", @"");

  [self setupForCallWithDirection:self.call.direction];
}

- (void)viewWillAppear:(BOOL)animated {
  [super viewWillAppear:animated];
  self.remoteUsername.text = [self.call remoteUserId];
}

- (void)setupForCallWithDirection:(SINCallDirection)direction {
  if (direction == SINCallDirectionIncoming) {
    // Enter the view from CallKit incoming call
    if ([self.callKitMediator callExistsWithId:self.call.callId]) {
      // Enter from CallKit lock screen
      if ([self.call state] == SINCallStateEstablished) {
        [self startCallDurationTimerWithSelector:@selector(onDurationTimer:)];
        [self showButtons:kButtonsHangup];
      } else {
        // Enter from CallKit but not lock screen
        [self setCallStatusText:@""];
        [self showButtons:kButtonsWakenByCallKit];
      }
    } else {
      // Enter the view from normal incoming call
      NSAssert(NO, @"Setup UI for normal incoming call not possible with CallKit. Call should exist in registry.", @"");
    }
  } else {  // Outgoing
    [self setCallStatusText:@"calling..."];
    [self showButtons:kButtonsHangup];
  }
}

#pragma mark - Call Actions

- (IBAction)hangup:(id)sender {
  [self.callKitMediator endCall:self.call];
}

- (void)onDurationTimer:(NSTimer *)unused {
  NSInteger duration = (NSInteger)[[NSDate date] timeIntervalSinceDate:[[self.call details] establishedTime]];
  [self setDuration:duration];
}

#pragma mark - SINCallDelegate

- (void)callDidProgress:(id<SINCall>)call {
  [self setCallStatusText:@"initiating..."];
}

- (void)callDidRing:(id<SINCall>)call {
  [self setCallStatusText:@"ringing..."];
  [[self.client audioController] startPlayingSoundFile:[self pathForSound:@"ringback.wav"] loop:YES error:nil];
}

- (void)callDidAnswer:(id<SINCall>)call {
  [self setCallStatusText:@"connecting..."];
  [[self.client audioController] stopPlayingSoundFile];
}

- (void)callDidEstablish:(id<SINCall>)call {
  [self startCallDurationTimerWithSelector:@selector(onDurationTimer:)];
  [self showButtons:kButtonsHangup];
}

- (void)callDidEnd:(id<SINCall>)call {
  [self.callKitMediator removeObserver:self];

  [self stopCallDurationTimer];
  [[self.client audioController] stopPlayingSoundFile];
  [self dismiss];
}

#pragma mark - Sounds

- (NSString *)pathForSound:(NSString *)soundName {
  return [[[NSBundle mainBundle] resourcePath] stringByAppendingPathComponent:soundName];
}

@end
