#import <OSLog/OSLog.h>
#import <Sinch/Sinch.h>
#import "AppDelegate.h"
#import "SINClientMediator.h"
#import "LoginViewController.h"
#import "CallViewController.h"

static os_log_type_t osLogTypeFrom(SINLogSeverity severity);

@interface AppDelegate () <SINManagedPushDelegate, SINClientMediatorDelegate>
@property (nonatomic, readwrite, strong) SINManagedPush *push;
@property (nonatomic, readwrite, strong) SINClientMediator *callKitMediator;
@property (nonatomic, readwrite, strong) os_log_t customLog;
@property (nonatomic, readonly) BOOL supportsVideo;
@end

@implementation AppDelegate

- (BOOL)application:(UIApplication *)application didFinishLaunchingWithOptions:(nullable NSDictionary *)launchOptions {
  self.customLog = os_log_create("com.sinch.sdk.app", "AppDelegate");
  os_log(self.customLog, "didFinishLaunchingWithOptions:");

  [Sinch setLogCallback:^(SINLogSeverity severity, NSString *area, NSString *message, NSDate *timestamp) {
    os_log_t logt = os_log_create("com.sinch.sdk.app", area.UTF8String);
    os_log_with_type(logt, osLogTypeFrom(severity), "%{public}@", message);
  }];

  /**
   * NOTE: You must specify proper environment depending on the signing entitlements
   * @see SINAPSEnvironment description
   */
  self.push = [Sinch managedPushWithAPSEnvironment:SINAPSEnvironmentDevelopment];
  self.push.delegate = self;
  [self.push setDesiredPushType:SINPushTypeVoIP];

  self.callKitMediator = [[SINClientMediator alloc] initWithDelegate:self supportsVideo:self.supportsVideo];

  if ([self.window.rootViewController isKindOfClass:LoginViewController.class]) {
    LoginViewController *lvc = (LoginViewController *)self.window.rootViewController;
    lvc.callKitMediator = self.callKitMediator;
  }

  return YES;
}

- (void)applicationWillEnterForeground:(UIApplication *)application {
  id<SINCall> call = self.callKitMediator.currentCall;
  if (call != nil) {
    [self transitionToCallView:call];
  } else {
    os_log_error(self.customLog, "applicationWillEnterForeground: No call to preset");
  }
}

#pragma mark -

- (void)handlePushNotification:(NSDictionary *)userInfo {
  os_log(self.customLog, "handlePushNotification -> createClientIfNeeded()");

  [self.callKitMediator createClientIfNeeded];

  NSAssert(self.callKitMediator.client != nil, @"Client at this point must always exist", @"");

  [self.callKitMediator.client relayPushNotification:userInfo];
}

- (void)transitionToCallView:(id<SINCall>)call {
  // If there is one established or initiating call, show the callView of the
  // current call when the App is brought to foreground. This is mainly to handle
  // the UI transition when clicking the App icon on the lockscreen CallKit UI,
  // and the UI transition when an incoming call is answered from homescreen CallKit UI.

  UIViewController *top = self.window.rootViewController;
  while (top.presentedViewController) {
    top = top.presentedViewController;
  }

  // When entering the application via the App button on the CallKit lockscreen,
  // and unlocking the device by PIN code/Touch ID, applicationWillEnterForeground:
  // will be invoked twice, and "top" will be CallViewController already after
  // the first invocation.
  if (![top isMemberOfClass:[CallViewController class]]) {
    [top performSegueWithIdentifier:@"callView" sender:call];
  }
}

- (BOOL)supportsVideo {
  NSNumber *video = [NSBundle.mainBundle objectForInfoDictionaryKey:@"SINAppSupportsVideo"];
  return video != nil && video.boolValue;
}

#pragma mark - SINManagedPushDelegate

- (void)managedPush:(SINManagedPush *)managedPush
    didReceiveIncomingPushWithPayload:(NSDictionary *)payload
                              forType:(NSString *)pushType {
  os_log(self.customLog, "didReceiveIncomingPushWithPayload: %{public}@", payload.description);

  // Since iOS 13 the application must report an incoming call to CallKit if a
  // VoIP push notification was used, and this must be done within the same run
  // loop as the push is received (i.e. GCD async dispatch must not be used).
  // See https://developer.apple.com/documentation/pushkit/pkpushregistrydelegate/2875784-pushregistry .
  [self.callKitMediator
      reportIncomingCallWithPushPayload:payload
                             completion:^(NSError *error) {
                               dispatch_async(dispatch_get_main_queue(), ^{
                                 [self handlePushNotification:payload];
                               });

                               if (error != nil) {
                                 os_log_error(self.customLog, "Error when reportintg call to CallKit: %{public}@",
                                              error.localizedDescription);
                               }
                             }];
}

#pragma mark - SINCallKitMediatorDelegate

- (void)handleIncomingCall:(id<SINCall>)call {
  [self transitionToCallView:call];
}

@end

static os_log_type_t osLogTypeFrom(SINLogSeverity severity) {
  switch (severity) {
    case SINLogSeverityInfo:
      return OS_LOG_TYPE_DEFAULT;
    case SINLogSeverityCritical:
      return OS_LOG_TYPE_FAULT;
    case SINLogSeverityWarn:
      return OS_LOG_TYPE_DEFAULT;
    case SINLogSeverityTrace:
      return OS_LOG_TYPE_DEBUG;
    default:
      return OS_LOG_TYPE_DEFAULT;
  }
}
