#include "SINJWT.h"
#include <CommonCrypto/CommonCrypto.h>

@interface NSString (SINJWTAdditions)
- (NSString*)replace:(NSString*)target with:(NSString*)replacement;
@end

static NSString* sin_JWT(NSDictionary* header, NSDictionary* payload, NSData* signingKey);
static NSData* sin_JSONSerialize(NSDictionary*);
static NSData* sin_HMAC_SHA256(NSData* key, NSString* message);
static NSString* sin_jwt_base64Encode(NSData*);
static NSString* sin_jwt_formatDate(NSDate*);
static NSData* sin_jwt_deriveSigningKey(NSString* applicationSecret, NSDate* issuedAt);
static NSString* sin_UTF8String(NSData*);
static NSData* sin_UTF8Encode(NSString*);

@implementation SINJWT

+ (NSString*)jwtForUserRegistrationWithApplicationKey:(NSString*)applicationKey
                                    applicationSecret:(NSString*)applicationSecret
                                               userId:(NSString*)userId
                                                nonce:(NSString*)nonce
                                             issuedAt:(NSDate*)issuedAt
                                             expireAt:(NSDate*)expireAt
                                     instanceExpireAt:(NSDate*)instanceExpireAt {
  NSDictionary* header = @{
    @"alg" : @"HS256",
    @"typ" : @"JWT",
    @"kid" : [NSString stringWithFormat:@"hkdfv1-%@", sin_jwt_formatDate(issuedAt)]
  };

  NSDictionary* payload = @{
    @"iss" : [NSString stringWithFormat:@"//rtc.sinch.com/applications/%@", applicationKey],
    @"sub" : [NSString stringWithFormat:@"//rtc.sinch.com/applications/%@/users/%@", applicationKey, userId],
    @"iat" : @((int)[issuedAt timeIntervalSince1970]),
    @"exp" : @((int)[expireAt timeIntervalSince1970]),
    @"sinch:rtc:instance:exp" : @((int)[instanceExpireAt timeIntervalSince1970]),
    @"nonce" : nonce
  };

  NSData* signingKey = sin_jwt_deriveSigningKey(applicationSecret, issuedAt);
  return sin_JWT(header, payload, signingKey);
}

+ (NSString*)jwtForUserRegistrationWithApplicationKey:(NSString*)applicationKey
                                    applicationSecret:(NSString*)applicationSecret
                                               userId:(NSString*)userId
                                                nonce:(NSString*)nonce
                                             issuedAt:(NSDate*)issuedAt
                                             expireAt:(NSDate*)expireAt {
  NSDictionary* header = @{
    @"alg" : @"HS256",
    @"typ" : @"JWT",
    @"kid" : [NSString stringWithFormat:@"hkdfv1-%@", sin_jwt_formatDate(issuedAt)]
  };

  NSDictionary* payload = @{
    @"iss" : [NSString stringWithFormat:@"//rtc.sinch.com/applications/%@", applicationKey],
    @"sub" : [NSString stringWithFormat:@"//rtc.sinch.com/applications/%@/users/%@", applicationKey, userId],
    @"iat" : @((int)[issuedAt timeIntervalSince1970]),
    @"exp" : @((int)[expireAt timeIntervalSince1970]),
    @"nonce" : nonce
  };

  NSData* signingKey = sin_jwt_deriveSigningKey(applicationSecret, issuedAt);
  return sin_JWT(header, payload, signingKey);
}

+ (NSString*)jwtForUserRegistrationWithApplicationKey:(NSString*)applicationKey
                                    applicationSecret:(NSString*)applicationSecret
                                               userId:(NSString*)userId {
  NSDate* now = [NSDate date];
  return [[self class] jwtForUserRegistrationWithApplicationKey:applicationKey
                                              applicationSecret:applicationSecret
                                                         userId:userId
                                                          nonce:[[NSUUID UUID] UUIDString]
                                                       issuedAt:now
                                                       expireAt:[now dateByAddingTimeInterval:(600)]];
}

@end

NSString* sin_jwt_formatDate(NSDate* date) {
  NSDateFormatter* formatter = [[NSDateFormatter alloc] init];
  formatter.locale = [NSLocale localeWithLocaleIdentifier:@"en_US_POSIX"];
  formatter.dateFormat = @"yyyyMMdd";
  formatter.timeZone = [NSTimeZone timeZoneForSecondsFromGMT:0];
  return [formatter stringFromDate:date];
}

static NSData* _sin_HMAC_SHA256(NSData* key, NSData* message) {
  unsigned char buf[CC_SHA256_DIGEST_LENGTH];
  CCHmac(kCCHmacAlgSHA256, key.bytes, [key length], message.bytes, [message length], buf);
  return [NSData dataWithBytes:buf length:CC_SHA256_DIGEST_LENGTH];
}

NSData* sin_HMAC_SHA256(NSData* key, NSString* message) {
  return _sin_HMAC_SHA256(key, sin_UTF8Encode(message));
}

NSString* sin_jwt_base64Encode(NSData* data) {
  // JWT RFC mandates URL-safe base64-encoding without padding.
  NSString* base64 = sin_UTF8String([data base64EncodedDataWithOptions:0]);
  return [[[base64 replace:@"+" with:@"-"] replace:@"/" with:@"_"] replace:@"=" with:@""];
}

NSData* sin_JSONSerialize(NSDictionary* payload) {
  NSData* json = [NSJSONSerialization dataWithJSONObject:payload options:0 error:nil];
  return [[sin_UTF8String(json) replace:@"\\/" with:@"/"] dataUsingEncoding:NSUTF8StringEncoding];
}

NSString* sin_JWT(NSDictionary* header, NSDictionary* payload, NSData* signingKey) {
  NSString* headerDotPayload =
      [@[ sin_jwt_base64Encode(sin_JSONSerialize(header)), sin_jwt_base64Encode(sin_JSONSerialize(payload)) ]
          componentsJoinedByString:@"."];

  NSString* signature = sin_jwt_base64Encode(sin_HMAC_SHA256(signingKey, headerDotPayload));

  return [@[ headerDotPayload, signature ] componentsJoinedByString:@"."];
}

NSData* sin_jwt_deriveSigningKey(NSString* applicationSecret, NSDate* issuedAt) {
  return sin_HMAC_SHA256([[NSData alloc] initWithBase64EncodedString:applicationSecret options:0],
                         sin_jwt_formatDate(issuedAt));
}

NSString* sin_UTF8String(NSData* data) {
  return [[NSString alloc] initWithData:data encoding:NSUTF8StringEncoding];
}

NSData* sin_UTF8Encode(NSString* s) {
  return [s dataUsingEncoding:NSUTF8StringEncoding];
}

@implementation NSString (SINJWTAdditions)
- (NSString*)replace:(NSString*)target with:(NSString*)replacement {
  return [self stringByReplacingOccurrencesOfString:target withString:replacement];
}
@end
