#import <Foundation/Foundation.h>
#import <OSLog/OSLog.h>
#import <Sinch/Sinch.h>

NS_ASSUME_NONNULL_BEGIN

@protocol SINClientMediatorDelegate <NSObject>
- (void)handleIncomingCall:(id<SINCall>)call;
@end

@protocol SINClientMediatorObserver <SINCallDelegate>
@end

typedef void (^VoidCompletion)(void);
typedef void (^ErrorCompletion)(NSError *_Nullable);
typedef void (^CallStartedCompletion)(id<SINCall> _Nullable, NSError *_Nullable);

/// This class shows one of the ways of Sinch SDK intagration with CallKit
@interface SINClientMediator : NSObject

/// Currently active call. Nil if there is no current call
@property (nonatomic, readonly, strong) _Nullable id<SINCall> currentCall;
/// Logged in instance of sinch client. nil if no instance was created
@property (nonatomic, readonly, strong) _Nullable id<SINClient> client;

- (instancetype)initWithDelegate:(id<SINClientMediatorDelegate>)delegate supportsVideo:(BOOL)supportsVideo;

/// Call to create client if needed using previously stored credentials/username
- (void)createClientIfNeeded;

/// Call to create sinch client providing userId
/// @param userId userId with which to create sinch client
/// @param completion Blcok to be called when client was create or error happened
- (void)createClientWithUserId:(NSString *)userId completion:(ErrorCompletion)completion;

/// Call to perform client logout
/// @param completion Block to be called when logout is completed
- (void)logout:(VoidCompletion)completion;

/// Call to report incoming call to CallKit.
/// @param payload Push paload received using PushKit
/// @param completion Blcok to be called when call is reported
- (void)reportIncomingCallWithPushPayload:(NSDictionary *)payload completion:(ErrorCompletion)completion;

/// Call to start outgoing call
/// @param destination Remote userId
/// @param completion Blcok to be called if call transaction successfully executed with CallKit
- (void)startOutgoingCallTo:(NSString *)destination completion:(CallStartedCompletion)completion;

/// Call to end current outgoing call
/// @param call Call to end
- (void)endCall:(id<SINCall>)call;

/// Call t check if there is a call with provided callId
/// @param callId for requested call
- (BOOL)callExistsWithId:(NSString *)callId;

/// Use this method to add call event observer
/// @param observer  Object conforming to
- (void)addObserver:(id<SINClientMediatorObserver>)observer;

/// Use this method to stop listening for call events
/// @param observer Object previously pased to addObserver:
- (void)removeObserver:(id<SINClientMediatorObserver>)observer;

@end

NS_ASSUME_NONNULL_END
