#import <Foundation/Foundation.h>
#import <CallKit/CallKit.h>
#import "SINClientMediator+Private.h"
#import "Sinch/SINCallDetails.h"

static CXCallEndedReason SINGetCallEndedReason(SINCallEndCause cause);

@implementation SINClientMediator (SINCallDelegate)

- (void)callDidProgress:(id<SINCall>)call {
  NSUUID *uuid = [self.callRegistry callKitUUIDForSinchId:call.callId];
  if (uuid != nil && call.direction == SINCallDirectionOutgoing) {
    [self.provider reportOutgoingCallWithUUID:uuid startedConnectingAtDate:call.details.startedTime];
  }

  [self fanoutDelegateCallWithCallback:^(id observer) {
    if ([observer respondsToSelector:@selector(callDidProgress:)]) {
      [observer callDidProgress:call];
    }
  }];
}

- (void)callDidRing:(id<SINCall>)call {
  [self fanoutDelegateCallWithCallback:^(id observer) {
    if ([observer respondsToSelector:@selector(callDidProgress:)]) {
      [observer callDidRing:call];
    }
  }];
}

- (void)callDidAnswer:(id<SINCall>)call {
  NSUUID *uuid = [self.callRegistry callKitUUIDForSinchId:call.callId];
  if (uuid != nil && call.direction == SINCallDirectionOutgoing) {
    [self.provider reportOutgoingCallWithUUID:uuid connectedAtDate:call.details.establishedTime];
  }

  [self fanoutDelegateCallWithCallback:^(id observer) {
    if ([observer respondsToSelector:@selector(callDidAnswer:)]) {
      [observer callDidAnswer:call];
    }
  }];
}

- (void)callDidEstablish:(id<SINCall>)call {
  [self fanoutDelegateCallWithCallback:^(id observer) {
    if ([observer respondsToSelector:@selector(callDidEstablish:)]) {
      [observer callDidEstablish:call];
    }
  }];
}

// Handle cancel/hangup event initiated by either caller or callee
- (void)callDidEnd:(id<SINCall>)call {
  call.delegate = nil;

  NSUUID *uuid = [self.callRegistry callKitUUIDForSinchId:call.callId];
  if (uuid != nil) {
    // Report end of the call to CallKit
    [self.provider reportCallWithUUID:uuid
                          endedAtDate:call.details.endedTime
                               reason:SINGetCallEndedReason(call.details.endCause)];
  }

  [self.callRegistry removeSinchCallWithId:call.callId];

  if (call.details.endCause == SINCallEndCauseError) {
    os_log_error(self.customLog,
                 "didEndCall with reason: %{public}d, error: %{public}@, Removing sinch call with ID: %{public}@",
                 (int)call.details.endCause, call.details.error == nil ? @"?" : call.details.error.localizedDescription,
                 call.callId);
  } else {
    os_log(self.customLog, "didEndCall with reason: %{public}d, Removing sinch call with ID: %{public}@",
           (int)call.details.endCause, call.callId);
  }

  [self fanoutDelegateCallWithCallback:^(id observer) {
    if ([observer respondsToSelector:@selector(callDidEnd:)]) {
      [observer callDidEnd:call];
    }
  }];
}

- (void)callDidAddVideoTrack:(id<SINCall>)call {
  [self fanoutDelegateCallWithCallback:^(id observer) {
    if ([observer respondsToSelector:@selector(callDidAddVideoTrack:)]) {
      [observer callDidAddVideoTrack:call];
    }
  }];
}

- (void)callDidPauseVideoTrack:(id<SINCall>)call {
  [self fanoutDelegateCallWithCallback:^(id observer) {
    if ([observer respondsToSelector:@selector(callDidPauseVideoTrack:)]) {
      [observer callDidPauseVideoTrack:call];
    }
  }];
}

- (void)callDidResumeVideoTrack:(id<SINCall>)call {
  [self fanoutDelegateCallWithCallback:^(id observer) {
    if ([observer respondsToSelector:@selector(callDidResumeVideoTrack:)]) {
      [observer callDidResumeVideoTrack:call];
    }
  }];
}

- (void)call:(id<SINCall>)call didEmitCallQualityWarningEvent:(id<SINCallQualityWarningEvent>)event {
  os_log(self.customLog, "Call Quality Event: %{public}@ %{public}@", event.name,
         event.type == SINCallQualityWarningEventTypeTrigger ? @"triggered" : @"recovered");
}

@end

static CXCallEndedReason SINGetCallEndedReason(SINCallEndCause cause) {
  switch (cause) {
    case SINCallEndCauseError:
      return CXCallEndedReasonFailed;
    case SINCallEndCauseDenied:
      return CXCallEndedReasonRemoteEnded;
    case SINCallEndCauseHungUp:
      // This mapping is not really correct, as SINCallEndCauseHungUp is the end cause also when the local peer ended
      // the call.
      return CXCallEndedReasonRemoteEnded;
    case SINCallEndCauseTimeout:
      return CXCallEndedReasonUnanswered;
    case SINCallEndCauseCanceled:
      return CXCallEndedReasonUnanswered;
    case SINCallEndCauseNoAnswer:
      return CXCallEndedReasonUnanswered;
    case SINCallEndCauseOtherDeviceAnswered:
      return CXCallEndedReasonAnsweredElsewhere;
    case SINCallEndCauseInactive:
      // This mapping is not really correct, as SINCallEndInactive is triggered by the local peer.
      return CXCallEndedReasonRemoteEnded;
    case SINCallEndCauseVoIPCallDetected:
      return CXCallEndedReasonFailed;
    case SINCallEndCauseGSMCallDetected:
      return CXCallEndedReasonFailed;
    default:
      break;
  }
  return CXCallEndedReasonFailed;
}
