#import <CallKit/CallKit.h>
#import "SINClientMediator.h"
#import "SINClientMediator+Private.h"
#import "Config.h"

NSString *const userIDKey = @"com.sinch.userId";

#pragma mark - SINCallClientDelegate
@interface SINClientMediator (SINCallClientDelegate) <SINCallClientDelegate>
@end

#pragma mark - SINClientDelegate
@interface SINClientMediator (SINClientDelegate) <SINClientDelegate>
@end

@implementation SINClientMediator

- (instancetype)initWithDelegate:(id<SINClientMediatorDelegate>)delegate supportsVideo:(BOOL)supportsVideo {
  self = [super init];

  if (self != nil) {
    self.delegate = delegate;
    self.customLog = os_log_create("com.sinch.sdk.app", "SINCallKitMediator");
    self.callRegistry = [[SINCallRegistry alloc] init];

    CXProviderConfiguration *config = [[CXProviderConfiguration alloc] initWithLocalizedName:@"Sinch"];
    config.maximumCallGroups = 1;
    config.maximumCallsPerCallGroup = 1;
    config.ringtoneSound = @"incoming.wav";
    config.supportedHandleTypes = [NSSet setWithArray:@[ @(CXHandleTypeGeneric) ]];
    config.supportsVideo = supportsVideo;

    self.acDelegate = [[AudioControllerDelegate alloc] init];
    self.isVideo = supportsVideo;
    self.callController = [[CXCallController alloc] init];
    self.provider = [[CXProvider alloc] initWithConfiguration:config];
    [self.provider setDelegate:self queue:nil];
    _observers = [NSPointerArray weakObjectsPointerArray];
  }

  return self;
}

- (void)createClientIfNeeded {
  if (self.client == nil) {
    NSString *userId = [[NSUserDefaults standardUserDefaults] objectForKey:userIDKey];
    if (userId == nil) {
      os_log_error(self.customLog, "No saved userId to create SinchClient");
    } else {
      [self createClientWithUserId:userId
                        completion:^(NSError *_Nullable error) {
                          if (error != nil) {
                            os_log_error(self.customLog, "SinchClient started with error: %{public}@",
                                         error.localizedDescription);
                          } else {
                            os_log(self.customLog, "SinchClient started successfully: (version:%{public}@)",
                                   [Sinch versionString]);
                          }
                        }];
    }
  }
}

- (void)reportIncomingCallWithPushPayload:(NSDictionary *)payload completion:(void (^)(NSError *error))completion {
  id<SINNotificationResult> notification = [SINManagedPush queryPushNotificationPayload:payload];

  if (notification.isCall) {
    id<SINCallNotificationResult> callNotification = notification.callResult;

    if ([self.callRegistry callKitUUIDForSinchId:callNotification.callId] == nil) {
      [self reportNewIncomingCallToCallKitWithNotification:callNotification completion:completion];
    }
  }
}

- (void)createClientWithUserId:(NSString *)userId completion:(void (^)(NSError *))completion {
  NSError *error;
  if (self.client == nil) {
    self.client = [Sinch clientWithApplicationKey:APPLICATION_KEY
                                  environmentHost:ENVIRONMENT_HOST
                                           userId:userId
                                            error:&error];
    if (!self.client && completion) {
      completion(error);
      return;
    }

    self.client.delegate = self;
    self.client.callClient.delegate = self;
    self.client.audioController.delegate = self.acDelegate;

    [self.client enableManagedPushNotifications];
    self.clientCreatedCallback = completion;

    [self.client start];
  } else {
    if (completion && self.client.isStarted) {
      completion(nil);
    }
  }
}
- (void)logout:(VoidCompletion)completion {
  if (self.client != nil) {
    if (self.client.isStarted) {
      // Remove push registration from Sinch backend
      [self.client unregisterPushNotificationDeviceToken];
      [self.client terminateGracefully];
    }

    self.client = nil;
  }

  if (completion != nil) {
    completion();
  }
}

- (void)startOutgoingCallTo:(NSString *)destination completion:(CallStartedCompletion)completion {
  CXHandle *handle = [[CXHandle alloc] initWithType:CXHandleTypeGeneric value:destination];
  CXStartCallAction *initiateCallAction = [[CXStartCallAction alloc] initWithCallUUID:[NSUUID UUID] handle:handle];
  initiateCallAction.video = self.isVideo;

  CXTransaction *transaction = [[CXTransaction alloc] initWithAction:initiateCallAction];
  self.callStartedCallback = completion;

  [self.callController requestTransaction:transaction
                               completion:^(NSError *_Nullable error) {
                                 if (error != nil) {
                                   os_log_error(self.customLog, "Error requesting start call transaction: %{public}@",
                                                error.localizedDescription);
                                   dispatch_async(dispatch_get_main_queue(), ^{
                                     if (completion) {
                                       completion(nil, error);
                                     }
                                     self.callStartedCallback = nil;
                                   });
                                 }
                               }];
}

- (void)endCall:(id<SINCall>)call {
  NSUUID *uuid = [self.callRegistry callKitUUIDForSinchId:call.callId];
  if (uuid == nil) {
    return;
  }

  CXEndCallAction *endCallAction = [[CXEndCallAction alloc] initWithCallUUID:uuid];
  CXTransaction *transaction = [[CXTransaction alloc] initWithAction:endCallAction];

  [self.callController requestTransaction:transaction
                               completion:^(NSError *_Nullable error) {
                                 if (error != nil) {
                                   os_log_error(self.customLog, "Error requesting end call transaction: %{public}@",
                                                error.localizedDescription);
                                 }
                                 self.callStartedCallback = nil;
                               }];
}

- (BOOL)callExistsWithId:(NSString *)callId {
  return [self.callRegistry sinchCallForCallId:callId] != nil;
}

/**
 * Returns currect active call if htere is one
 */
- (id<SINCall>)currentCall {
  NSArray<id<SINCall>> *calls = self.callRegistry.activeSinchCalls;

  if (calls.count > 0) {
    id<SINCall> call = calls[0];
    if (call.state == SINCallStateInitiating || call.state == SINCallStateEstablished ||
        call.state == SINCallStateProgressing || call.state == SINCallStateRinging) {
      return call;
    }
  }
  return nil;
}

#pragma mark - Implementation

- (void)reportNewIncomingCallToCallKitWithNotification:(id<SINCallNotificationResult>)notification
                                            completion:(ErrorCompletion)completion {
  NSUUID *uuid = [NSUUID UUID];

  [self.callRegistry mapCallKitUUID:uuid toSinchCallId:notification.callId];

  os_log(self.customLog, "reportNewIncomingCallToCallKit: ckid:%{public}@ callId:%{public}@", uuid.UUIDString,
         notification.callId);

  CXCallUpdate *update = [[CXCallUpdate alloc] init];
  update.remoteHandle = [[CXHandle alloc] initWithType:CXHandleTypeGeneric value:notification.remoteUserId];
  update.hasVideo = notification.isVideoOffered;

  [self.provider reportNewIncomingCallWithUUID:uuid
                                        update:update
                                    completion:^(NSError *_Nullable error) {
                                      if (error != nil) {
                                        // If we get an error here from the OS, it is possibly the callee's phone has
                                        // "Do Not Disturb" turned on, check CXErrorCodeIncomingCallError in CXError.h
                                        [self hangupCallOnErrorWithCallId:notification.callId];
                                      }
                                      BOOL hasActiveCallKitCalls = self.callRegistry.activeSinchCalls.count > 0;
                                      completion(error);
                                      // As per Apple docs we are forced to call `CXProvider.reportNewIncomingCall`
                                      // whenever a new VoIP push arrives. However Sinch SDK can handle only a single
                                      // active call at a time (relaying any new incoming call push, when there's
                                      // already ongoing call will cause the new call to be automatically denied and
                                      // `SinchCallClientDelegate.didReceiveIncomingCall` will not be called). In order
                                      // to prevent CallKit UI from showing the new call data, we need to mark it as
                                      // ended as soon as completion block is invoked.
                                      if (hasActiveCallKitCalls) {
                                        [self.provider reportCallWithUUID:uuid
                                                              endedAtDate:nil
                                                                   reason:CXCallEndedReasonDeclinedElsewhere];
                                      }
                                    }];
}

- (void)hangupCallOnErrorWithCallId:(NSString *)callId {
  id<SINCall> call = [self.callRegistry sinchCallForCallId:callId];
  if (call == nil) {
    os_log_error(self.customLog, "Unable to find sinch call for callId: %{public}@", callId);
    return;
  }

  [call hangup];
  [self.callRegistry removeSinchCallWithId:callId];
}

- (void)fanoutDelegateCallWithCallback:(void (^)(id observer))callback {
  NSArray *observerObjects = [self.observers allObjects];
  for (id observer in observerObjects) {
    if (observer) {
      callback(observer);
    }
  }
}

- (void)addObserver:(id<SINClientMediatorObserver>)observer {
  NSArray *observerObjects = [self.observers allObjects];
  if ([observerObjects containsObject:observer])
    return;

  [self.observers addPointer:(__bridge void *)observer];
}

- (void)removeObserver:(id<SINClientMediatorObserver>)observer {
  for (NSUInteger i = 0; i < self.observers.count; ++i) {
    if ([self.observers pointerAtIndex:i] == (__bridge void *)observer) {
      [self.observers removePointerAtIndex:i];
      break;
    }
  }
}

@end
