package com.sinch.android.rtc.sample.push

import android.app.ProgressDialog
import android.content.Intent
import android.os.Bundle
import android.util.Log
import android.view.KeyEvent
import android.view.inputmethod.EditorInfo
import android.widget.Button
import android.widget.EditText
import android.widget.TextView
import android.widget.Toast
import com.google.android.gms.tasks.Task
import com.google.firebase.messaging.FirebaseMessaging
import com.sinch.android.rtc.*
import com.sinch.android.rtc.sample.push.FcmTokenPersistence.getFcmRegistrationToken
import com.sinch.android.rtc.sample.push.FcmTokenPersistence.setFcmRegistrationToken
import com.sinch.android.rtc.sample.push.JWT.create
import java.io.IOException

class LoginActivity : BaseActivity(), SinchService.StartFailedListener,
    PushTokenRegistrationCallback, UserRegistrationCallback {

    private lateinit var loginButton: Button
    private lateinit var loginNameEditText: EditText
    private var progressSpinner: ProgressDialog? = null
    private var userId: String? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.login)
        loginNameEditText = findViewById(R.id.loginName)
        loginButton = findViewById(R.id.loginButton)

        loginNameEditText.setOnEditorActionListener { _: TextView?, actionId: Int, _: KeyEvent? ->
            if (actionId == EditorInfo.IME_ACTION_DONE) {
                loginClicked()
                return@setOnEditorActionListener true
            }
            false
        }
        loginButton.apply {
            isEnabled = false
            setOnClickListener { loginClicked() }
        }
        acquireFcmRegistrationToken()
    }

    override fun onServiceConnected() {
        if (sinchServiceInterface?.isStarted == true) {
            openPlaceCallActivity()
        } else {
            sinchServiceInterface?.setStartListener(this)
        }
    }

    override fun onPause() {
        dismissSpinner()
        super.onPause()
    }

    override fun onFailed(error: SinchError) {
        Toast.makeText(this, error.toString(), Toast.LENGTH_LONG).show()
        dismissSpinner()
    }

    override fun onStarted() {
        openPlaceCallActivity()
    }

    private fun acquireFcmRegistrationToken() {
        FirebaseMessaging.getInstance().token.addOnCompleteListener { task: Task<String?> ->
            if (task.isSuccessful) {
                loginButton.isEnabled = true
                setFcmRegistrationToken(this, task.result)
            } else {
                Toast.makeText(this, "Couldn't acquire FCM registration token", Toast.LENGTH_LONG)
                    .show()
                Log.e(TAG, "Error while getting FCM registration token", task.exception)
            }
        }
    }

    private fun startClientAndOpenPlaceCallActivity() {
        // Start Sinch Client, it'll result onStarted() callback from where the place call
        // activity will be started
        if (sinchServiceInterface?.isStarted == false) {
            sinchServiceInterface?.startClient()
            showSpinner()
        }
    }

    private fun loginClicked() {
        val username = loginNameEditText.text.toString()
        sinchServiceInterface?.username = username
        if (username.isEmpty()) {
            Toast.makeText(this, "Please enter a name", Toast.LENGTH_LONG).show()
            return
        }

        /**
         * For the login flow we showcase UserController usage here. It's a lightweight component
         * that can be use to register the user for receiving push notifications without the need
         * to initialize the entire SinchClient.
         *
         * SinchClient provides exactly same functionality when started.
         *
         */
        userId = username
        val userController = try {
            UserController.builder()
                .context(applicationContext)
                .applicationKey(SinchService.APP_KEY)
                .userId(userId.orEmpty())
                .environmentHost(SinchService.ENVIRONMENT)
                .pushConfiguration(
                    PushConfiguration.fcmPushConfigurationBuilder()
                        .senderID(SinchService.APP_FCM_SENDER_ID)
                        .registrationToken(getFcmRegistrationToken(this)!!)
                        .build()
                )
                .build()
        } catch (e: IOException) {
            Log.e(TAG, "Error while building user controller", e)
            return
        }
        userController.registerUser(this, this)
    }

    private fun openPlaceCallActivity() {
        startActivity(Intent(this, PlaceCallActivity::class.java))
    }

    private fun showSpinner() {
        progressSpinner = ProgressDialog(this).apply {
            setTitle("Logging in")
            setMessage("Please wait...")
            show()
        }
    }

    private fun dismissSpinner() {
        progressSpinner?.dismiss()
    }

    override fun onUserRegistrationFailed(error: SinchError) {
        dismissSpinner()
        Toast.makeText(this, "Registration failed!", Toast.LENGTH_LONG).show()
    }

    override fun onUserRegistered() {
        // Instance is registered, but we'll wait for another callback, assuring that the push
        // token is registered as well, meaning we can receive incoming calls.
    }

    override fun onPushTokenRegistered() {
        dismissSpinner()
        startClientAndOpenPlaceCallActivity()
    }

    override fun onPushTokenRegistrationFailed(sinchError: SinchError) {
        dismissSpinner()
        Toast.makeText(
            this, "Push token registration failed - incoming calls can't be received!",
            Toast.LENGTH_LONG
        ).show()
    }

    // The most secure way is to obtain the credentials from the backend,
    // since storing the Application Secret in the app is not safe.
    // Following code demonstrates how the JWT that serves as credential should be created,
    // provided the Application Key (APP_KEY), Application Secret (APP_SECRET) and User ID.
    override fun onCredentialsRequired(clientRegistration: ClientRegistration) {
        // NB: This implementation just emulates what should be an async procedure, with JWT
        // .create() being run on your backend.
        clientRegistration.register(
            create(
                SinchService.APP_KEY,
                SinchService.APP_SECRET,
                userId.orEmpty()
            )
        )
    }

    companion object {
        private val TAG = LoginActivity::class.java.simpleName
    }
}