package com.sinch.android.rtc.sample.hmspush

import android.app.NotificationManager
import android.content.Intent
import android.os.Bundle
import android.util.Log
import android.view.View
import android.widget.Button
import android.widget.TextView
import com.sinch.android.rtc.calling.Call
import com.sinch.android.rtc.calling.CallListener

class IncomingCallScreenActivity : BaseActivity() {

    private lateinit var answerButton: Button
    private lateinit var declineButton: Button
    private lateinit var remoteUserTextView: TextView

    private val callId: String get() = intent.getStringExtra(SinchService.CALL_ID).orEmpty()
    private val call: Call? get() = sinchServiceInterface?.getCall(callId)
    private val audioPlayer: AudioPlayer by lazy {
        AudioPlayer(this)
    }
    private val action: String? get() = intent.action

    private val mClickListener = View.OnClickListener { view ->
        when (view.id) {
            R.id.answerButton -> answerClicked()
            R.id.declineButton -> declineClicked()
            else -> {}
        }
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.incoming)
        answerButton = findViewById(R.id.answerButton)
        declineButton = findViewById(R.id.declineButton)
        remoteUserTextView = findViewById(R.id.remoteUser)

        answerButton.setOnClickListener(mClickListener)
        declineButton.setOnClickListener(mClickListener)
        audioPlayer.playRingtone()
    }

    override fun onResume() {
        super.onResume()
        val id = intent.getIntExtra(EXTRA_ID, -1)
        if (id > 0) {
            val notificationManager =
                getSystemService(NOTIFICATION_SERVICE) as NotificationManager
            notificationManager.cancel(id)
        }
    }

    override fun onServiceConnected() {
        val call = call
        if (call != null) {
            call.addCallListener(SinchCallListener())
            remoteUserTextView.text = call.remoteUserId
            if (ACTION_ANSWER == action) {
                answerClicked()
            } else if (ACTION_IGNORE == action) {
                declineClicked()
            }
        } else {
            Log.e(TAG, "Started with invalid callId, aborting")
            finish()
        }
    }

    private fun answerClicked() {
        val call = call
        audioPlayer.stopRingtone()
        if (call != null) {
            Log.d(TAG, "Answering call")
            call.answer()
            val intent = Intent(this, CallScreenActivity::class.java)
                .apply {
                    putExtra(SinchService.CALL_ID, callId)
                }
            startActivity(intent)
        } else {
            finish()
        }
    }

    private fun declineClicked() {
        audioPlayer.stopRingtone()
        call?.hangup()
        finish()
    }

    private inner class SinchCallListener : CallListener {

        override fun onCallEnded(call: Call) {
            val cause = call.details.endCause
            Log.d(TAG, "Call ended, cause: $cause")
            audioPlayer.stopRingtone()
            finish()
        }

        override fun onCallEstablished(call: Call) {
            Log.d(TAG, "Call established")
        }

        override fun onCallProgressing(call: Call) {
            Log.d(TAG, "Call progressing")
        }
    }

    companion object {
        private val TAG: String = IncomingCallScreenActivity::class.java.simpleName

        const val ACTION_ANSWER = "answer"
        const val ACTION_IGNORE = "ignore"
        const val EXTRA_ID = "id"
        const val MESSAGE_ID = 14
    }
}